/*  Copyright (C) 2001    Dobin Rutishauser        dobin.rutishauser@gmx.net
                          Bjrn Paetzel                       kolrabi@gmx.de
   
  ------------------------------------------------------------------------

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation;  either  version 2 of the  License, or
    any later version.

    This program is distributed in the hope that it  will be useful, but
    WITHOUT  ANY  WARRANTY;  without   even  the  implied   warranty  of
    MERCHANTABILITY  or  FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU
    General Public License for more details.
	
    You should have received a copy  of the  GNU  General Public License
    along with this program (GPL.TXT); if not, write to the 

	Free Software Foundation, Inc., 
	59 Temple Place, Suite 330, 
	Boston, MA
	02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <libnet.h>
#include <pcap.h>

#include "../common/messages.h"
#include "../common/config.h"
#include "../common/utils.h"

#include "serversockets.h"
#include "serverconfig.h"
#include "scan.h"


/* pcap while alarm timeout flag */
unsigned int pcap_flag = 0;


/* 
 * alarm handler
 *
 * set timeout flag for pcap
 *
*/

int alarm_handler(int signo)
{
	pcap_flag = 0;
	return(0);
}


/* recv packet
 *
 * receive packets and decide wether our port is open or closed
*/

int recvpacket(struct _infopacket *infopacket, struct itimerval *tmr)
{
struct pcap_pkthdr usefull;
u_char *packet;
struct ip *ippkt;
struct tcphdr *tcppkt;

	struct ether_header {
        u_char ether_dhost[6]; /* destination host */
        u_char ether_shost[6]; /* source host */
        u_short ether_type;    /* frame type */
	} *ethpkt;

	if (setitimer (ITIMER_REAL, tmr, NULL) == -1) {
		logprintf (LVL_WARNING, "could not set timer");
		return(0);
	}

	pcap_flag = 1;

   while (pcap_flag == 1)
   {
      packet = (u_char *) pcap_next(libpcap_socket, &usefull);
      if (packet != NULL)
      {
			
			ethpkt = (struct ether_header *)(packet);
			if (ntohs (ethpkt->ether_type) != ETHERTYPE_IP) continue;
		
			ippkt = (struct ip*)(packet + LIBNET_ETH_H);
			if (ippkt->ip_p != 6) continue;
			
         tcppkt = (struct tcphdr*)(packet + LIBNET_ETH_H + LIBNET_IP_H);

			if (infopacket->dst_ip == ippkt->ip_src.s_addr && 
				infopacket->dst_port == ntohs(tcppkt->th_sport) && 
				infopacket->seq_nr+1 == ntohl(tcppkt->th_ack))
			{
				if (infopacket->mode == 1) {   /* SYN SCAN */
					if ((tcppkt->th_flags & TH_SYN) && (tcppkt->th_flags & TH_ACK))
					{
						return (PORT_OPEN);
					}

					if (tcppkt->th_flags & TH_RST)
					{
						return (PORT_CLOSED);
					}
				} 
				else
				{  /* NULL FIN XMAS ACK SCAN */

					if (tcppkt->th_flags & TH_RST)
					{
						return(PORT_CLOSED);
					} 
				}
			}
      }
   }

	/* timeout */
	if (infopacket->mode == 1) { /* syn scan port timed out */
		return(PORT_CLOSED);
	} else {		/* timeout on null fin xmas ack scan means open port */
		return(PORT_OPEN);
	}

}


/*
 * send packet
 *
 * send a packet with specific header to the destination
 *
*/

int sendpacket(struct _infopacket *infopacket)
{
	static char localhostname[128];
	int packet_size, c;
	u_char *packet;
	u_long src_ip;
	u_char flags;

	/* coz i don't want for every packet a localhost name resolution, i use
	   a static variable.. */
	if (strlen (localhostname) < 1)
	{
		gethostname(localhostname, sizeof(localhostname));
	}

	if (!(src_ip = libnet_name_resolve(localhostname, LIBNET_RESOLVE))) 
	{ 
		libnet_error(LIBNET_ERR_FATAL, "Bad source IP address: %s\n", optarg);
		return(0); 
	} 

	packet_size = LIBNET_IP_H + LIBNET_TCP_H; 

	libnet_init_packet(packet_size, &packet); 
	if (packet == NULL) 
	{ 
		libnet_error(LIBNET_ERR_FATAL, "libnet_init_packet failed\n"); 
	} 

	/* build IP Header */
	libnet_build_ip(LIBNET_TCP_H,
			IPTOS_LOWDELAY, 	/* tos, change? */
			(u_short)(random()), /* id field */
			0, 
			(u_char)(random()),	/* ttl */
			IPPROTO_TCP,
			src_ip, 
			infopacket->dst_ip, 
			NULL, 	/* payload */
			0,
			packet);

	/* build TCP Header */
	switch(infopacket->mode)
	{
		case 0:
			flags = TH_URG;
			break;
		case 1:
			flags = TH_SYN;
			break;
		case 2:
			flags = TH_FIN;
			break;
		case 3:
			flags = TH_FIN | TH_PUSH | TH_URG;
			break;
		case 4:
			flags = TH_ACK;
			break;
	}

	libnet_build_tcp(
		infopacket->src_port, 
		infopacket->dst_port, 
		infopacket->seq_nr, 
		(u_long)(random()), /* ack */
		flags, 
		(u_short)(random()), /* window */
		0,
		NULL,
		0,
		packet + LIBNET_IP_H);

	if (libnet_do_checksum(packet, IPPROTO_TCP, LIBNET_TCP_H) == -1) 
	{ 
		libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n"); 
		return(0);
	} 

	c = libnet_write_ip(libnet_socket, packet, packet_size); 
	if (c < packet_size) { 
		libnet_error(LN_ERR_WARNING, "libnet_write_ip only wrote %d bytes\n", c); 
		libnet_destroy_packet(&packet); 
		return(0);
	} 
  
	libnet_destroy_packet(&packet); 

return (1); 
} 



/*
 * stealth scan
 *
 * stealthscan handler
 *
*/

int stealthscan (struct _infopacket *infopacket) 
{
int res=0;
struct itimerval tmr;

	infopacket->src_port = (u_short)(random()); /* create source port */
   infopacket->seq_nr = (u_long)(random()); /* seq number */

	/* sux... i must convert mikrosekonds to seconds!? */
	if (infopacket->timeout > 9) {
		tmr.it_value.tv_sec = infopacket->timeout / 10;
		tmr.it_value.tv_usec = (infopacket->timeout - ((infopacket->timeout / 10) * 10)); /* hmm */
	} else {
		tmr.it_value.tv_sec = 0;
		tmr.it_value.tv_usec = infopacket->timeout * 100000;
	}
	tmr.it_interval.tv_sec = 0;
	tmr.it_interval.tv_usec = 0;

	/* send the packet/scan the port */
   res = sendpacket(infopacket);
	if (res < 1) {
		logprintf (LVL_WARNING, "could not send packet" );
		return (PORT_UNKNOWN);
	}

	/* receive the answer */
   res = recvpacket(infopacket, &tmr);

return(res);
}


/*
 * connect scan
 *
 * connects to name and port
 * return PORT_OPEN or PORT_CLOSED
 *
*/

int connectscan (struct _infopacket *infopacket)
{
	SOCKET			scansocket;
	SOCKADDR_IN		saddress;

	if ((scansocket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		perror("socket");
		exit(1);
	}

	/* set address and port */
	saddress.sin_family			= AF_INET;
	saddress.sin_port				= htons(infopacket->dst_port);
	saddress.sin_addr.s_addr	= infopacket->dst_ip;
	bzero(&(saddress.sin_zero), 8);

	if (connect (scansocket, (struct sockaddr *)&saddress, 
		sizeof(struct sockaddr)) == -1)
	{
		shutdown (scansocket,0);
		return (PORT_CLOSED);
	} else {
		shutdown (scansocket,0);
		return (PORT_OPEN);
	}

}


/*
 * scan()
 *
 * return values
 * PORT_OPEN or PORT_CLOSED or PORT_UNKNOWN
 *
 * decide which scan method to use
 * 
*/

int scan (struct _infopacket *infopacket)
{
	printf("."); fflush(stdout);

	switch(infopacket->mode)
	{
		case	SCAN_CONNECT:
			return connectscan(infopacket);
			break;

		case	SCAN_SYN:
			return stealthscan(infopacket);
			break;

		case	SCAN_FIN:
			return stealthscan(infopacket);
			break;
			
		case	SCAN_XMAS:
			return stealthscan(infopacket);
			break;
			
		case	SCAN_NULL:
			return stealthscan(infopacket);
			break;
			
		case	SCAN_ACK:
			return stealthscan(infopacket);
			break;
	}
	
	return PORT_UNKNOWN;
}
