/*
 * libOp
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: modsib-x86.c,v 1.2 2002/04/24 21:04:19 cosine Exp $
 */

/*
 * Some general info on ModR/M and SIB bytes to remind myself
 * from time to time:
 *
 * A ModR/M byte is constructed as follows:
 *
 * Bits: 7 6           5 4 3         2 1 0
 *       Mod           Reg/Opcode    R/M
 *
 * Mod        = Combines with the R/M field
 *
 * Reg/Opcode = Either a register number or three more bits
 *              of opcode information
 *
 * R/M        = Either a register or combines with Mod to form
 *              an effective address
 *
 * A SIB byte is constructed in the following manner:
 *
 * Bits:  7 6          5 4 3         2 1 0
 *        Scale        Index         Base
 *
 * Scale = A number (0-3) that you must raise 2 to in order to
 *         determine the scaling factor
 *
 * Index = A number (0-7) which tells you the effective address
 *         of the SIB byte. I have placed these values in SibTable[].
 *
 * Base  = A number (0-7) which tells you the register to use as
 *         the base register. I have placed these registers in
 *         SibBaseRegisters[].
 */

#include <assert.h>

#include "modsib-x86.h"
#include "prefix-x86.h"

/*
 * These are possible values for intel ModR/M bytes (16 bit
 * addressing). See the Intel Architecture Software Developer's
 * Manual Vol. 2, sections 2.4 - 2.6
 */
static struct ModSib ModRMTable16[NUMVALUES] = {
  { M_BX_SI, 0x00, 0 },
  { M_BX_DI, 0x01, 0 },
  { M_BP_SI, 0x02, 0 },
  { M_BP_DI, 0x03, 0 },
  { M_SI, 0x04, 0 },
  { M_DI, 0x05, 0 },
  { M_BP, 0x06, BITS16 },
  { M_BX, 0x07, 0 },
  { -1, -1, -1 }
};

/*
 * These are possible values for ModR/M bytes with 32 bit addressing
 */
static struct ModSib ModRMTable32[NUMVALUES] = {
  { M_EAX, 0x00, 0 },
  { M_ECX, 0x01, 0 },
  { M_EDX, 0x02, 0 },
  { M_EBX, 0x03, 0 },
  { M_NONE, 0x04, SIBBYTE },
  { M_EBP, 0x05, BITS32 },
  { M_ESI, 0x06, 0 },
  { M_EDI, 0x07, 0 },
  { -1, -1, -1 }
};

static struct ModSib SibTable[NUMVALUES] = {
  { M_EAX, 0x00, 0 },
  { M_ECX, 0x01, 0 },
  { M_EDX, 0x02, 0 },
  { M_EBX, 0x03, 0 },
  { M_NONE, 0x04, 0 },
  { M_EBP, 0x05, 0 },
  { M_ESI, 0x06, 0 },
  { M_EDI, 0x07, 0 },
  { -1, -1, -1 }
};

/*
 * Address offsets corresponding to ModR/M and SIB bytes
 */
char *ModSibOffsets[] = {
  "ah",
  "al",
  "ax",
  "bh",
  "bl",
  "bp",
  "bx",
  "bp+di",
  "bp+si",
  "bx+di",
  "bx+si",
  "ch",
  "cl",
  "cx",
  "dh",
  "di",
  "dl",
  "dx",
  "eax",
  "ebp",
  "ebx",
  "ecx",
  "edi",
  "edx",
  "esi",
  "esp",
  "mm0",
  "mm1",
  "mm2",
  "mm3",
  "mm4",
  "mm5",
  "mm6",
  "mm7",
  "si",
  "sp",
  "xmm0",
  "xmm1",
  "xmm2",
  "xmm3",
  "xmm4",
  "xmm5",
  "xmm6",
  "xmm7",
  "",       /* M_NONE */
};

/*
 * SIB base registers - the register's index in this array
 * corresponds to it's base value 0-7
 */
char *SibBaseRegisters[] = {
  "eax",
  "ecx",
  "edx",
  "ebx",
  "esp",
  "ebp",
  "esi",
  "edi"
};

/*
FindModSib()
  Called from FindOpcode() to determine if a prospective match
meets the ModR/M and SIB byte requirements.

Inputs: data   - opcode data string
        opptr  - prospective opcode match
        bytes  - number of opcode bytes matched so far in 'data'
        modptr - where to store ModR/M match if any
        sibptr - where to store SIB match if any

Return: 1 if a successful modrm/sib match was found
        -1 if unsuccessful

Side effects: if a match is found, modptr and sibptr are modified
              to point to their respective data structures
*/

int
FindModSib(unsigned char *data, struct OpCode *opptr,
           unsigned int bytes, struct ModSib **modptr,
           struct ModSib **sibptr)

{
  unsigned char modrm;   /* ModR/M byte */
  unsigned char mod,     /* Mod portion of ModR/M byte */
                reg,     /* REG portion of ModR/M byte */
                rm;      /* R/M portion of ModR/M byte */

  assert(data && opptr && modptr && sibptr);

  modrm = *(data + bytes);
  reg = (modrm >> 3) & 0x07;

  if ((opptr->digit != REGRM) && (reg != opptr->digit))
  {
    /*
     * The opcode's modr/m digit specification does not
     * fit with this particular modr/m byte - bad match
     */
    return (-1);
  }

  rm = modrm & 0x07;
  *modptr = GetModAddress(rm, PrefixFlags);
  assert(*modptr != 0);

  /*
   * Some ModRM bytes require additional addressing information
   * stored in a SIB byte directly after the ModRM byte.
   */
  mod = modrm >> 6;
  if (((*modptr)->flags & SIBBYTE) && (mod != 3))
  {
    unsigned char sibbyte; /* SIB byte */
    unsigned char index;   /* index portion of SIB byte */

    sibbyte = (unsigned char) *(data + bytes + 1);

    /*
     * All we need from sibbyte are the bits 3,4,5 to
     * determine the effective address - strip out the
     * other bits
     */
    index = (sibbyte >> 3) & 0x07;

    *sibptr = GetSibAddress(index);
    assert(*sibptr != 0);
  } /* if ((*modptr)->flags & SIBBYTE) */

  return (1);
} /* FindModSib() */

/*
GetModAddress()
  Look up the value of a modrm byte in ModRMTable??[] and determine
it's effective address offset

Inputs: rm     - "R/M" field of ModR/M byte
        pflags - Prefix flags (PX_xxx)
*/

struct ModSib *
GetModAddress(unsigned char rm, unsigned int pflags)

{
  struct ModSib *mptr;

  if (pflags & PX_32BITADDR)
    mptr = ModRMTable32;
  else
    mptr = ModRMTable16;

  while (mptr->index != (-1))
  {
    if (mptr->value == rm)
      return (mptr);

    ++mptr;
  }

  return (0);
} /* GetModAddress() */

/*
GetSibAddress()
  Determine the effective address offset given the index portion
of a SIB byte

Inputs: index - index bytes of SIB byte

Return: pointer to ModSib structure containing 'index'
*/

struct ModSib *
GetSibAddress(unsigned char index)

{
  struct ModSib *mptr;

  mptr = SibTable;

  while (mptr->index != (-1))
  {
    if (mptr->value == index)
      return (mptr);

    ++mptr;
  }

  return (0);
} /* GetSibAddress() */

/*
FindModRegister()
  Locate the appropriate ModR/M register corresponding to "rm"
and with a size given by "operand"

Inputs: reg     - "REG" field of ModR/M byte
        operand - operand attributes (size)

Return: index of Registers[] corresponding to the correct register
*/

int
FindModRegister(unsigned char reg, unsigned int operand)

{
  switch (reg)
  {
    case 0:
    {
      if (operand & BITS8)
        return (R_AL);
      else if (operand & BITS16)
        return (R_AX);
      else if (operand & BITS32)
        return (R_EAX);
      else if (operand & REG_MMX)
        return (R_MM0);
      else if (operand & REG_XMM)
        return (R_XMM0);

      break;
    } /* case '0' */

    case 1:
    {
      if (operand & BITS8)
        return (R_CL);
      else if (operand & BITS16)
        return (R_CX);
      else if (operand & BITS32)
        return (R_ECX);
      else if (operand & REG_MMX)
        return (R_MM1);
      else if (operand & REG_XMM)
        return (R_XMM1);

      break;
    }

    case 2:
    {
      if (operand & BITS8)
        return (R_DL);
      else if (operand & BITS16)
        return (R_DX);
      else if (operand & BITS32)
        return (R_EDX);
      else if (operand & REG_MMX)
        return (R_MM2);
      else if (operand & REG_XMM)
        return (R_XMM2);

      break;
    }

    case 3:
    {
      if (operand & BITS8)
        return (R_BL);
      else if (operand & BITS16)
        return (R_BX);
      else if (operand & BITS32)
        return (R_EBX);
      else if (operand & REG_MMX)
        return (R_MM3);
      else if (operand & REG_XMM)
        return (R_XMM3);

      break;
    }

    case 4:
    {
      if (operand & BITS8)
        return (R_AH);
      else if (operand & BITS16)
        return (R_SP);
      else if (operand & BITS32)
        return (R_ESP);
      else if (operand & REG_MMX)
        return (R_MM4);
      else if (operand & REG_XMM)
        return (R_XMM4);

      break;
    }

    case 5:
    {
      if (operand & BITS8)
        return (R_CH);
      else if (operand & BITS16)
        return (R_BP);
      else if (operand & BITS32)
        return (R_EBP);
      else if (operand & REG_MMX)
        return (R_MM5);
      else if (operand & REG_XMM)
        return (R_XMM5);

      break;
    }

    case 6:
    {
      if (operand & BITS8)
        return (R_DH);
      else if (operand & BITS16)
        return (R_SI);
      else if (operand & BITS32)
        return (R_ESI);
      else if (operand & REG_MMX)
        return (R_MM6);
      else if (operand & REG_XMM)
        return (R_XMM6);

      break;
    }

    case 7:
    {
      if (operand & BITS8)
        return (R_BH);
      else if (operand & BITS16)
        return (R_DI);
      else if (operand & BITS32)
        return (R_EDI);
      else if (operand & REG_MMX)
        return (R_MM7);
      else if (operand & REG_XMM)
        return (R_XMM7);

      break;
    }
  } /* switch (reg) */

  /*
   * We should not get here
   */
  return (-1);
} /* FindModRegister() */

/*
FindModRegisterByRM()
  This routine is called when we have an opcode specified with a /digit between 0
and 7. In this case only the R/M field of the ModR/M byte is used. The REG field
will contain the digit number. This routine will look up which register the R/M
field corresponds to.

Inputs: rm      - "R/M" field of ModR/M byte
        operand - operand attributes (size)

Return: index of Registers[] corresponding to the correct register
*/

int
FindModRegisterByRM(unsigned char rm, unsigned int operand)

{
  switch (rm)
  {
    case 0:
    {
      if (operand & BITS8)
        return (R_AL);
      else if (operand & BITS16)
        return (R_AX);
      else if (operand & BITS32)
        return (R_EAX);
      else if (operand & REG_MMX)
        return (R_MM0);
      else if (operand & REG_XMM)
        return (R_XMM0);

      break;
    } /* case '0' */

    case 1:
    {
      if (operand & BITS8)
        return (R_CL);
      else if (operand & BITS16)
        return (R_CX);
      else if (operand & BITS32)
        return (R_ECX);
      else if (operand & REG_MMX)
        return (R_MM1);
      else if (operand & REG_XMM)
        return (R_XMM1);

      break;
    }

    case 2:
    {
      if (operand & BITS8)
        return (R_DL);
      else if (operand & BITS16)
        return (R_DX);
      else if (operand & BITS32)
        return (R_EDX);
      else if (operand & REG_MMX)
        return (R_MM2);
      else if (operand & REG_XMM)
        return (R_XMM2);

      break;
    }

    case 3:
    {
      if (operand & BITS8)
        return (R_BL);
      else if (operand & BITS16)
        return (R_BX);
      else if (operand & BITS32)
        return (R_EBX);
      else if (operand & REG_MMX)
        return (R_MM3);
      else if (operand & REG_XMM)
        return (R_XMM3);

      break;
    }

    case 4:
    {
      if (operand & BITS8)
        return (R_AH);
      else if (operand & BITS16)
        return (R_SP);
      else if (operand & BITS32)
        return (R_ESP);
      else if (operand & REG_MMX)
        return (R_MM4);
      else if (operand & REG_XMM)
        return (R_XMM4);

      break;
    }

    case 5:
    {
      if (operand & BITS8)
        return (R_CH);
      else if (operand & BITS16)
        return (R_BP);
      else if (operand & BITS32)
        return (R_EBP);
      else if (operand & REG_MMX)
        return (R_MM5);
      else if (operand & REG_XMM)
        return (R_XMM5);

      break;
    }

    case 6:
    {
      if (operand & BITS8)
        return (R_DH);
      else if (operand & BITS16)
        return (R_SI);
      else if (operand & BITS32)
        return (R_ESI);
      else if (operand & REG_MMX)
        return (R_MM6);
      else if (operand & REG_XMM)
        return (R_XMM6);

      break;
    }

    case 7:
    {
      if (operand & BITS8)
        return (R_BH);
      else if (operand & BITS16)
        return (R_DI);
      else if (operand & BITS32)
        return (R_EDI);
      else if (operand & REG_MMX)
        return (R_MM7);
      else if (operand & REG_XMM)
        return (R_XMM7);

      break;
    }
  } /* switch (reg) */

  /*
   * We should not get here
   */
  return (-1);
} /* FindModRegisterByRM() */
