/*
 * libOp
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: prefix-x86.c,v 1.2 2001/04/17 00:26:36 cosine Exp $
 */

#include <assert.h>

#include "disasm-x86.h"
#include "prefix-x86.h"

/*
 * Global: Bitmask representing prefixes of the current opcode.
 */
unsigned int          PrefixFlags = 0;

/*
 * If any of these bytes follow a REP prefix (0xF3), it is a
 * REPE (repeat while equal) instruction.
 */
int PrefixREPE[] = {
  0xA6,
  0xA7,
  0xAE,
  0xAF,
  (-1)
};

/*
CheckPrefix()
  Check the prefix of an opcode to determine if any special
measures should be taken

Inputs: code  - opcode string
        flags - variable to contain PX_xxx bitmasks

Return: number of bytes in the prefix

Side effects: 'flags' is modified to reflect various prefix flags
*/

unsigned int
CheckPrefix(unsigned char *code, unsigned int *flags)

{
  unsigned int numbytes = 0;
  unsigned char byte;
  int done;

  assert(code && flags);

  done = 0;

  while (!done)
  {
    byte = *code++;

    switch (byte)
    {
      /*
       * Indicates operand size override. If in 16 bit mode,
       * instructions with 32 bit operands will be chosen over
       * those with 16 bit operands. If in 32 bit mode, instructions
       * with 16 bit operands will be chosen over those with 32
       * bit operands.
       */
      case 0x66:
      {
        *flags ^= PX_32BITDATA;
        ++numbytes;

        break;
      } /* case 0x66 */

      /*
       * Addressing size override. If in 16 bit mode, 32 bit addressing
       * will be used and vice versa.
       */
      case 0x67:
      {
        *flags ^= PX_32BITADDR;
        ++numbytes;

        break;
      } /* case 0x67 */

      /*
       * Indicates REP (repeat operation) prefix
       */
      case 0xF3:
      {
        int *tmp;
        unsigned int flag;

        /*
         * If we have an "F3 0F" situation, it is not a REP, but
         * a Katmai SSE extension
         */
        if (*code == 0x0F)
        {
          done = 1;
          break;
        }

        flag = PX_REP;
        for (tmp = PrefixREPE; *tmp != (-1); ++tmp)
        {
          if ((int) *code == *tmp)
            flag = PX_REPE;
        }

        *flags |= flag;
        ++numbytes;

        break;
      } /* case 0xF3: */

      /*
       * Indicates a REPNE (repeat while not equal) prefix
       */
      case 0xF2:
      {
        *flags |= PX_REPNE;
        ++numbytes;

        break;
      } /* case 0xF2: */

      /*
       * Segment overrides
       */
      case 0x2E:         /* cs */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_CS;
        ++numbytes;

        break;
      } /* case 0x2E: */

      case 0x36:         /* ss */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_SS;
        ++numbytes;

        break;
      } /* case 0x36: */

      case 0x3E:         /* ds */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_DS;
        ++numbytes;

        break;
      } /* case 0x3E: */

      case 0x26:         /* es */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_ES;
        ++numbytes;

        break;
      } /* case 0x26: */

      case 0x64:         /* fs */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_FS;
        ++numbytes;

        break;
      } /* case 0x64: */

      case 0x65:         /* gs */
      {
        *flags |= PX_SEGOVER;
        SegmentOverride = R_GS;
        ++numbytes;

        break;
      } /* case 0x65: */

      default:
      {
        done = 1;
        break;
      }
    } /* switch (byte) */
  } /* while (!done) */

  return (numbytes);
} /* CheckPrefix() */

/*
MatchPrefix()
  Called from FindOpcode() to determine if a potential matching
opcode meets the prefix requirements.

Inputs: ptr   - OpCode pointer which is a potential match
        flags - prefix flags that 'ptr' is required to pass

Return: 2 if 'ptr' passes prefix requirements
        1 if 'ptr' has no operands and there is a 32 bit prefix
        0 if not a good match
*/

int
MatchPrefix(struct OpCode *ptr, unsigned int flags)

{
  assert(ptr != 0);

  if (flags & PX_32BITDATA)
  {
    /*
     * This opcode had a 32 bit indicator in it's prefix - make
     * sure an operand is expecting 32 bits. If the operands
     * are not expecting 32 bits, it may still be acceptable
     * if there is no 32 bit form of the instruction, in which
     * case having a 0x66 prefix is pointless - so return 1
     * to indicate a low priority match.
     */
    if (!(ptr->operands[0] & BITS32) &&
        !(ptr->operands[1] & BITS32) &&
        !(ptr->operands[2] & BITS32))
      return (1);
  } /* if (flags & PX_32BITDATA) */
  else if (ptr->operands[0] == BITS32)
  {
    /*
     * There is no 32 bit prefix, yet this opcode is marked
     * as expecting 32 bits with no operands - this is not
     * a good match.
     * Opcodes which have their first operand equal to BITS32
     * are marked as "void32" in optab-x86.dat.
     *
     * Note: this is a rare case: such as INSD, IRETD etc.
     *       (opcodes with no operands and 32 bit addressing)
     */
    return (0);
  }

  /*
   * Passed all the tests
   */
  return (2);
} /* MatchPrefix() */
