/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_disassemble.c,v 1.17 2002/04/11 01:26:20 cosine Exp $
 */

#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <assert.h>

#include "alddefs.h"
#include "defs.h"
#include "disassemble.h"
#include "file.h"
#include "load.h"
#include "main.h"
#include "msg.h"
#include "scroll.h"
#include "print.h"
#include "window.h"

#include "libDebug.h"

/*
 * libOp includes
 */
#include "disasm.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static void ResetDisassemblyPosition(struct aldWorkspace *ws);

/*
c_disassemble()
 Disassemble the current file

Format for this command:
  disassemble [start [stop]] [-num <number>] [-section <name>]
*/

void
c_disassemble(struct aldWorkspace *ws, int ac, char **av)

{
  unsigned char *data;  /* data to disassemble */
  unsigned char *membuf;
  long insnum,          /* number of instructions to disassemble */
       inscnt;          /* number of instructions disassembled so far */
  char *endptr;
  long length;
  int cnt,
      alen;
  unsigned int start,
               end;
  unsigned int address; /* address of instruction to be disassembled */
  char *section;        /* are we disassembling a specific section? */
  long ndumped,         /* number of bytes dumped */
       numbytes;        /* total bytes */
  unsigned char spill[MAXOPLEN + 1];
  int gotstart;         /* did we get a starting address? */

  start = end = 0;
  gotstart = 0;
  section = 0;
  insnum = 0;
  ndumped = numbytes = 0;

  for (cnt = 1; cnt < ac; ++cnt)
  {
    alen = strlen(av[cnt]);
    if (!Strncasecmp(av[cnt], "-section", alen))
    {
      if (++cnt >= ac)
      {
        Print(ws, P_COMMAND, "No section name specified");
        return;
      }

      /*
       * Check if they already specified a starting address
       */
      if (start)
        continue;

      if (!FindSectionByName(ws->loadWorkspace_p, av[cnt], &start, &end))
      {
        Print(ws, P_COMMAND, "No such section: %s", av[cnt]);
        return;
      }

      section = av[cnt];
    }
    else if (!Strncasecmp(av[cnt], "-num", alen))
    {
      if (++cnt >= ac)
      {
        Print(ws, P_COMMAND, "No instruction count specified");
        return;
      }

      insnum = strtol(av[cnt], &endptr, 0);
      if ((endptr == av[cnt]) || (*endptr != '\0'))
      {
        Print(ws, P_COMMAND, MSG_INVNUM, av[cnt]);
        return;
      }
    }
    else
    {
      /*
       * It must be the starting or ending address
       */
      if (!start)
      {
        start = strtoul(av[cnt], &endptr, 0);
        gotstart = 1;
      }
      else if (!end)
        end = strtoul(av[cnt], &endptr, 0);

      if ((endptr == av[cnt]) || (*endptr != '\0'))
      {
        Print(ws, P_COMMAND, MSG_INVNUM, av[cnt]);
        return;
      }

      /*
       * Start/End addresses take priority over -section
       */
      section = 0;
    }
  }

  if (section)
  {
    /*
     * We are disassembling a specific section - set
     * ObjectFileOffset to the correct position.
     */
    setLoadedVirtualFileOffset(ws->loadWorkspace_p, start);

    Print(ws, P_COMMAND, "Disassembling section %s (0x%08X - 0x%08X)",
      section,
      start,
      end);

  #ifdef USE_CURSES

    /*
     * Clear any disassembly data from previous runs
     */
    if (ModeCurses)
      ClearScrollData(DisassemblyFrame);

  #endif /* USE_CURSES */
  } /* if (section) */

  membuf = 0;
  if (gotstart)
  {
    if (!end)
    {
      if (!insnum)
      {
        /*
         * They gave a starting memory address, but did not say when
         * to stop, so default to 15 instructions
         */
        insnum = 15;
      }

      /*
       * 150 bytes should be sufficient for 15 instructions
       */
      end = start + 150;
    }

    numbytes = end - start;
    if (numbytes < 0)
    {
      Print(ws, P_COMMAND, "Ending address is lower than starting address");
      return;
    }

    /*
     * Add fifteen to account for the very last instruction - in case
     * it starts at location 'end'
     */
    numbytes += 15;

    ndumped = dumpMemoryDebug(ws->debugWorkspace_p,
                              (char **) &membuf,
                              start,
                              numbytes);

    if (!ndumped)
    {
      Print(ws,
            P_COMMAND,
            MSG_NOACCESS,
            start,
            strerror(errno));

      if (membuf)
        MyFree(membuf);

      return;
    }

    /*
     * In case we couldn't access all the memory we tried to dump,
     * set 'end' correctly.
     */
    if (ndumped < numbytes)
      end = start + ndumped;

    data = membuf;
    address = start;
  } /* if (gotstart) */
  else
  {
    /*
     * We are using file offsets rather than memory locations
     */

    if (getLoadedFileOffset(ws->loadWorkspace_p) >=
        ws->loadWorkspace_p->MappedSize)
    {
      ResetDisassemblyPosition(ws);
    }

    data = (unsigned char *) ws->loadWorkspace_p->MapPtr;
    data += getLoadedFileOffset(ws->loadWorkspace_p);
    address = getLoadedVirtualFileOffset(ws->loadWorkspace_p);
  }

  inscnt = 0;

  startPrintBurst(ws->printWorkspace_p);

  /*
   * There are many conditions which could cause the loop to
   * stop, so I'm using an infinite loop with a bunch of 'break'
   * checks.
   */
  while (1)
  {
    /*
     * stop if we reach the instruction limit
     */
    if (insnum && (inscnt++ >= insnum))
      break;

    /*
     * stop if we hit the end of the address range we are
     * disassembling
     */
    if (gotstart)
    {
      if (end && (address > end))
        break;
    }
    else
    {
      /*
       * stop if we go past the end of the file
       */
      if (getLoadedFileOffset(ws->loadWorkspace_p) >=
          ws->loadWorkspace_p->MappedSize)
        break;
      else
      {
        /*
         * Check if we are nearing the end of the file - if so,
         * use a separate buffer rather than MapPtr, because
         * if our disassembling routine screws up, it may try to
         * read a byte or two past the end of MapPtr, causing
         * a crash.
         */
        if ((getLoadedFileOffset(ws->loadWorkspace_p) + MAXOPLEN) >=
            ws->loadWorkspace_p->MappedSize)
        {
          unsigned int ii;

          cnt = 0;
          memset((void *) spill, 0, sizeof(spill));

          for (ii = getLoadedFileOffset(ws->loadWorkspace_p); ii < ws->loadWorkspace_p->MappedSize; ++ii)
            spill[cnt++] = (unsigned char) *((unsigned char *) ws->loadWorkspace_p->MapPtr + ii);

          data = spill;
        }
      }
    }

    /*
     * stop if we hit the end of the section we are disassembling
     */
    if (section && (address >= end))
      break;

    /*
     * disassemble the next instruction
     */
    length = procDisassemble(ws->disassembleWorkspace_p,
                             data,
                             address,
                             P_DISASSEMBLY);

    assert(length != 0);

    if (length > 0)
    {
      data += length;
      address += length;

      if (!membuf)
        incLoadedVirtualFileOffset(ws->loadWorkspace_p, length);
    }
    else
      break; /* something went wrong */
  } /* while (1) */

  endPrintBurst(ws->printWorkspace_p);

  if (membuf)
  {
    MyFree(membuf);
    if (ndumped < numbytes)
    {
      Print(ws,
            P_COMMAND,
            MSG_NOACCESS,
            start + ndumped,
            strerror(errno));
    }
  }

#ifdef USE_CURSES

  if (ModeCurses)
  {
    /*
     * The above Print() statements did not actually output
     * anything to the window - they simply stored the data in
     * the dynamic pointer array DisassemblyFrame->data. We need
     * to do it this way so we can write the data now with the
     * top line highlighted, which DrawWindow() will take care of.
     */
    DrawWindow(DisassemblyFrame);
  }

#endif /* USE_CURSES */
} /* c_disassemble() */

/*
ResetDisassemblyPosition()
  Called when we disassemble the last instruction in the file -
reset the offsets and clear the disassembly window
*/

static void
ResetDisassemblyPosition(struct aldWorkspace *ws)

{
  Print(ws,
        P_COMMAND,
        "End of file reached, repositioning offset to 0x%08x",
        getLoadedVirtualEntryPoint(ws->loadWorkspace_p));

  setLoadedVirtualFileOffset(ws->loadWorkspace_p,
                             getLoadedVirtualEntryPoint(ws->loadWorkspace_p));

#ifdef USE_CURSES

  /*
   * Clear any disassembly data from previous runs
   */
  if (ModeCurses)
    ClearScrollData(DisassemblyFrame);

#endif /* USE_CURSES */
} /* ResetDisassemblyPosition() */
