/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: disassemble.c,v 1.13 2002/04/11 01:26:23 cosine Exp $
 */

#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "defs.h"
#include "disassemble.h"
#include "file.h"
#include "load.h"
#include "main.h"
#include "print.h"
#include "scroll.h"
#include "set.h"
#include "window.h"

#include "libDebug.h"

/*
 * libOp includes
 */
#include "disasm.h"

/*
 * libString includes
 */
#include "alloc.h"

#ifdef USE_CURSES

static void AddDisasmAddress(unsigned int address);
#if 0
static int FindDisasmAddress(unsigned int address);
#endif

#endif

#ifdef USE_CURSES

#if 0
/*
 * Local: start address of current section we are debugging through
 */
static unsigned int     CurrentSectionStart = 0;
#endif

#endif /* USE_CURSES */

/*
initDisassemble()
  Initialize a disassembly workspace

Return: pointer to workspace
*/

struct disassembleWorkspace *
initDisassemble()

{
  struct disassembleWorkspace *ws;

  ws = (struct disassembleWorkspace *) malloc(sizeof(struct disassembleWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initDisassemble: malloc failed: %s\n",
      strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct disassembleWorkspace));

  ws->NextInstruction = 0;
  ws->flags = 0;
  ws->CurrentSectionStart = 0;

  return (ws);
} /* initDisassemble() */

/*
termDisassemble()
  Terminate a disassembly workspace

Inputs: ws - workspace to terminate
*/

void
termDisassemble(struct disassembleWorkspace *ws)

{
  if (!ws)
    return;

  free(ws);
} /* termDisassemble() */

/*
procDisassemble()
  Backend for c_disassemble() - call the appropriate libOp
routines to disassemble instructions from the file we are
debugging (this routine disassembles exactly 1 instruction)

Inputs: data    - opcodes to disassemble
        address - memory address of opcodes
        owin    - window to print disassembled instruction to

Return: number of bytes disassembled
        -1 if error occurred
*/

long
procDisassemble(struct disassembleWorkspace *ws, unsigned char *data,
                unsigned int address, unsigned int owin)

{
  int ii;
  char buffer[MAXLINE];  /* buffer result is stored in */
  char mcode[MAXLINE];   /* buffer containing opcodes */
  int mlen;
  long length;           /* length of opcode we disassembled */

  *buffer = '\0';

  length = DisAssemble(data, buffer, ws->flags);
  if (length < 0)
  {
    Print(mainWorkspace_p,
          P_COMMAND,
          "Disassembly error: %s",
          buffer);
    return (-1);
  }

  if (length == 0)
  {
    /*
     * A length of 0 means we have encountered an unknown
     * instruction - output a message and keep going.
     */
    Print(mainWorkspace_p,
          P_COMMAND,
          "Unknown instruction at offset 0x%08x: 0x%02X",
          address,
          (unsigned char) *data);

    sprintf(mcode, "%02X", (unsigned char) *data++);
    sprintf(buffer, "???");

    /*
     * Set length to 1 to update the offsets in order to get to
     * the next instruction
     */
    length = 1;
  }
  else
  {
    /*
     * Now advance data 'length' bytes - also store the machine
     * code along the way
     */
    mlen = 0;
    for (ii = 0; ii < length; ++ii)
    {
      sprintf(mcode + mlen, "%02X", (unsigned char) *data++);
      mlen += 2;
    }
  }

  /*
   * Output disassembled instruction
   */
  Print(mainWorkspace_p,
        owin,
        "%08X  %-20s       %s",
        address,
        mcode,
        buffer);

#ifdef USE_CURSES

  if (ModeCurses)
  {
    if (owin == P_DEBUG)
      AddDisasmAddress(address);
  }

#endif

  return (length);
} /* procDisassemble() */

/*
DisplayNextInstruction()
  Called when the debugged process has stopped for whatever reason -
disassemble and output the next instruction to be executed.

Return: none
*/

void
DisplayNextInstruction(struct aldWorkspace *main_p)

{
  unsigned char *codeptr;
  long ndumped;

  main_p->disassembleWorkspace_p->NextInstruction = getAddressDebug(main_p->debugWorkspace_p);

  if (ModeConsole)
  {
    /*
     * If in console mode, we don't have a window of the
     * next instructions, so disassemble the very next
     * instruction and display it
     */

    fputc('\n', stdout);

    codeptr = 0;

    /*
     * 20 bytes should be more than ample to grap one opcode
     */
    ndumped = dumpMemoryDebug(main_p->debugWorkspace_p,
                              (char **) &codeptr,
                              main_p->disassembleWorkspace_p->NextInstruction,
                              20);

    assert(codeptr != 0);

    if (ndumped == 0)
    {
      Print(main_p,
            P_COMMAND,
            "Error disassembling next instruction (address: 0x%08lX)",
            main_p->disassembleWorkspace_p->NextInstruction);
    }
    else
    {
      procDisassemble(main_p->disassembleWorkspace_p,
                      codeptr,
                      main_p->disassembleWorkspace_p->NextInstruction,
                      P_DEBUG);
    }

    MyFree(codeptr);

    return;
  } /* if (ModeConsole) */

#if 0

#ifdef USE_CURSES

  if (ModeCurses)
  {
    unsigned int start,
                 end;
    unsigned int bytes;
    int ret;
    int insnidx; /* index of instruction */
    long length;

    ret = FindSectionByAddr(MapPtr, NextInstruction, &start, &end);
    if (ret)
    {
      /*
       * We found the section of our object file that the next
       * instruction resides in - if it is the same section
       * as the last instruction, do nothing, otherwise assume
       * we have entered a new section and disassemble and display
       * it.
       */
      if (start != CurrentSectionStart)
      {
        unsigned char *data;
        unsigned int curaddr;

        /*
         * We have jumped to a new section - clear the debug
         * window and disassemble/display the new section
         */
        ClearScrollData(DebugFrame);

        codeptr = 0;
        bytes = end - start;

        ndumped = DumpMemory((char **) &codeptr,
                             start,
                             bytes);

        assert(codeptr != 0);

        if (ndumped == bytes)
        {
          data = codeptr;
          curaddr = start;

          while (curaddr < end)
          {
            length = Disasm(data, curaddr, P_DEBUG);
            assert(length != 0);

            if (length > 0)
            {
              data += length;
              curaddr += length;
            }
            else
              break; /* something went wrong */
          }
        }
        else
        {
          Print(main_p,
                P_COMMAND,
                "Error disassembling next instruction (address: 0x%08lX)",
                NextInstruction);
        }

        MyFree(codeptr);

        CurrentSectionStart = start;
      }
    }
    else
    {
      fprintf(stdout, "SEGMENT NOT FOUND - WE ARE OUTSIDE FILE\n");
    }

    /*
     * Do the actual output to the window
     */
    insnidx = FindDisasmAddress(NextInstruction);
    assert(insnidx != (-1));

    SetCurrentElement(DebugFrame, insnidx);

    if (!(DebugFrame->flags & W_RAISED))
    {
      /*
       * The RefreshWindows() call after this won't update
       * the new highlight line which SetCurrentElement()
       * set if DebugFrame is not on top, so raise it and
       * lower it
       */
      ReRaiseFrame(DebugFrame);
    }
  } /* if (ModeCurses) */

#endif /* USE_CURSES */

#endif /* 0 */
} /* DisplayNextInstruction() */

#ifdef USE_CURSES

/*
AddDisasmAddress()
  Add given address to DebugFrame's address array

Inputs: address - instruction address

Return: none
*/

static void
AddDisasmAddress(unsigned int address)

{
  struct Frame *window;

  /*
   * This routine is called after AddScrollData(), so we can
   * assume that DataElements has been appropriately updated
   * already.
   */
  window = DebugFrame;
  if (!window->addresses)
  {
    /*
     * There is no data in the array - add this as the first
     * element
     */
    window->addresses = (unsigned int *) MyMalloc(sizeof(unsigned int));
    window->addresses[0] = address;
  }
  else
  {
    window->addresses = (unsigned int *) MyRealloc(window->addresses, sizeof(unsigned int) * window->DataElements);
    window->addresses[window->DataElements - 1] = address;
  }
} /* AddDisasmAddress() */

#if 0
/*
FindDisasmAddress()
  Attempt to locate a disassembly address in DebugFrame's
addresses array

Inputs: address - instruction address

Return: index of DebugFrame->address of address if found
        -1 if not
*/

static int
FindDisasmAddress(unsigned int address)

{
  int ii;

  for (ii = 0; ii < DebugFrame->DataElements; ++ii)
  {
    if (DebugFrame->addresses[ii] == address)
      return (ii);
  }

  return (-1);
} /* FindDisasmAddress() */

#endif /* 0 */

#endif /* USE_CURSES */
