/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: scroll.c,v 1.8 2002/04/11 01:26:23 cosine Exp $
 */

#include "defs.h"

#ifdef USE_CURSES

#include <string.h>
#include <assert.h>
#include <ncurses.h>

#include "alddefs.h"
#include "main.h"
#include "misc.h"
#include "print.h"
#include "scroll.h"
#include "window.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static void HighlightLine(struct Frame *frame, char *line);

/*
PageUp()
 Called when the user presses page-up in the specified window -
scroll up one page.
*/

void
PageUp(struct Frame *frame)

{
  assert(frame != 0);

  if (!frame->data)
    return; /* empty frame */

  if (frame->CurrentElement == 0)
  {
    debug("current pointer is 0, no going up\n");
    return; /* already at the top */
  }

  if (frame->TopElement < frame->lines)
  {
    debug("We're at the top screen, no going up\n");
    frame->TopElement = 0;
    frame->CurrentElement = 0;
  }
  else
  {
    frame->TopElement -= frame->lines;
    frame->CurrentElement -= frame->lines;
    debug("Paging up, top = %d, cur = %d\n",
      frame->TopElement,
      frame->CurrentElement);
  }

  DrawWindow(frame);
} /* PageUp() */

/*
PageDown()
 Called when the user presses page-down in the specified window -
scroll down one page.
*/

void
PageDown(struct Frame *frame)

{
  assert(frame != 0);

  if (!frame->data)
    return; /* nothing in the frame */

  assert(frame->CurrentElement < frame->DataElements);

  if (frame->CurrentElement == (frame->DataElements - 1))
    return; /* the cursor is at the bottom element - can't pagedown */

  if ((frame->TopElement + frame->lines) >= frame->DataElements)
  {
    /*
     * The last element is already on the page - don't touch
     * TopElement, but move CurrentElement to the bottom - only
     * if we are not in block mode though - in block mode, we
     * want to keep CurrentElement == TopElement.
     */
    debug("on the last page, no going down\n");
    if (!(frame->flags & W_BLOCKMODE))
      frame->CurrentElement = frame->DataElements - 1;
  }
  else
  {
    /*
     * There is at least 1 line off of the current window (page).
     * Advance TopElement to next page.
     */
    frame->TopElement += frame->lines;

    if ((frame->CurrentElement + frame->lines) >= frame->DataElements)
    {
      /*
       * There is not a full page left, so advancing CurrentElement
       * by frame->lines would move the cursor beyond the last
       * element - we could set it to the last element here, but I
       * think it looks better to set it to TopElement.
       */
      frame->CurrentElement = frame->TopElement;
    }
    else
      frame->CurrentElement += frame->lines;

    debug("going down, top=%d, cur=%d\n",
      frame->TopElement,
      frame->CurrentElement);
  }

  DrawWindow(frame);
} /* PageDown() */

/*
KeyUp()
 Called when the user presses the up arrow key in the specified
window - scroll the window up one line.
*/

void
KeyUp(struct Frame *frame)

{
  assert(frame != 0);

  if (!frame->data)
    return; /* empty frame */

  if (frame->CurrentElement == 0)
    return; /* it is already at the top */

  if (frame->flags & W_BLOCKMODE)
  {
    assert (frame->TopElement != 0);

    --frame->TopElement;
    --frame->CurrentElement;
    debug("KeyUp: shifting frame down 1, new TE = %d, CE = %d\n",
      frame->TopElement,
      frame->CurrentElement);
  }
  else
  {
    if (frame->CurrentElement > frame->TopElement)
    {
      --frame->CurrentElement;
      debug("scrolling current up 1, leaving top alone\n");
    }
    else if (frame->CurrentElement == frame->TopElement)
    {
      /*
       * The highlight (cursor) is on the top element of the screen,
       * scroll up one
       */
      assert(frame->TopElement != 0);
      assert(frame->CurrentElement != 0);

      --frame->TopElement;
      --frame->CurrentElement;
      debug("moving top and current up one\n");
    }
  } /* if (frame->flags & W_BLOCKMODE) */

  DrawWindow(frame);
} /* KeyUp() */

/*
KeyDown()
 Called when the user presses the down arrow key - scroll window or
cursor down one line
*/

void
KeyDown(struct Frame *frame)

{
  assert(frame != 0);

  if (!frame->data)
    return; /* empty frame */

  if (frame->CurrentElement == (frame->DataElements - 1))
    return; /* it is at the bottom of the frame elements */

  if (frame->flags & W_BLOCKMODE)
  {
    /*
     * When a frame is in block mode, a down arrow will shift
     * everything up one (ie: it shifts the whole text block up)
     */
    ++frame->TopElement;
    ++frame->CurrentElement;
    debug("KeyDown: shifting frame up 1, new TE = %d, CE = %d, TOTAL E = %d\n",
      frame->TopElement,
      frame->CurrentElement,
      frame->DataElements);
  }
  else
  {
    if (frame->CurrentElement == (frame->TopElement + frame->lines - 1))
    {
      /*
       * The cursor is on the very last line in the frame, move
       * TopElement down one.
       */
      ++frame->TopElement;
      ++frame->CurrentElement;
    }
    else
    {
      /*
       * The cursor is somewhere in the middle of the frame - leave
       * TopElement alone.
       */
      ++frame->CurrentElement;
    }
  } /* if (frame->flags & W_BLOCKMODE) */

  DrawWindow(frame);
} /* KeyDown() */

void
KeyHome(struct Frame *frame)

{
  assert (frame != 0);

  frame->TopElement = frame->CurrentElement = 0;

  DrawWindow(frame);
} /* KeyHome() */

/*
KeyEnd()
  Called when user presses the "end" key - advance frame to
the bottom.
*/

void
KeyEnd(struct Frame *frame)

{
  assert(frame != 0);

  frame->TopElement = frame->CurrentElement = frame->DataElements - 1;

  DrawWindow(frame);
} /* KeyEnd() */

/*
SetCurrentElement()
  Set the CurrentElement field of the given frame

Inputs: frame - scrolling frame
        index  - index of frame->data to set as new current element

Return: none
*/

void
SetCurrentElement(struct Frame *frame, int index)

{
  if (index < frame->lines)
    frame->TopElement = 0;
  else if (index >= (frame->TopElement + frame->lines))
    frame->TopElement = index;

  frame->CurrentElement = index;

  DrawWindow(frame);
} /* SetCurrentElement() */

/*
DrawWindow()
 Redraw a frame

Inputs: frame - frame to redraw
*/

void
DrawWindow(struct Frame *frame)

{
  int curidx; /* current index of frame->data[] */

  /*
   * When to stop printing (1 past the index of the last element
   * to be printed)
   */
  int end;

  assert(frame != 0);
  assert(frame->window != 0);

  /*
   * Clear frame
   */
  werase(frame->window);

  wmove(frame->window, 0, 0);

  curidx = frame->TopElement;

  if ((frame->TopElement + frame->lines) > frame->DataElements)
    end = frame->DataElements;
  else
    end = frame->TopElement + frame->lines;

  debug("DrawWindow(): TOP = %d, END = %d\n", curidx, end);

  /*
   * Write new data to frame
   */
  while (curidx < end)
  {
    if ((curidx == frame->CurrentElement) &&
        (frame->flags & W_HIGHLIGHT))
    {
      /*
       * We are about to draw the line which frame->CurrentElement
       * corresponds to - highlight it
       */
      HighlightLine(frame, frame->data[curidx]);
    }
    else
    {
      waddstr(frame->window, frame->data[curidx]);
      /*
       * Do not add a \n onto a window marked W_SCROLL if we
       * are on the last piece of data to be written, otherwise
       * there will always be a blank line at the bottom of the
       * window not being used
       */
      if (!(frame->flags & W_SCROLL) || (curidx != (end - 1)))
        waddch(frame->window, '\n');
    }

    ++curidx;
  }

  /*
   * Refresh window
   */
  RefreshFrames();
} /* DrawWindow() */

/*
AddScrollData()
 Add a line of data into a scrolling window's buffer

Inputs: frame - window to add data into
        data  - actual data to add

Return: none
*/

void
AddScrollData(struct Frame *frame, char *data)

{
  assert(frame != 0);
  assert(data != 0);

  if (!frame->data)
  {
    /*
     * There is no data in the array - add this as the first
     * element
     */
    frame->data = (char **) MyMalloc(sizeof(char *));
    frame->data[0] = Strdup(data);
    frame->DataElements = 1;
  }
  else
  {
    ++frame->DataElements;
    frame->data = (char **) MyRealloc(frame->data,
                               sizeof(char *) * frame->DataElements);
    frame->data[frame->DataElements - 1] = Strdup(data);
  }

  if (frame->flags & W_SCROLL)
  {
    int LastElement; /* index of last element in frame->data */

    /*
     * This frame scrolls up each time data is added to it
     * (to make room for the new line), so we must keep TopElement
     * and CurrentElement updated properly.
     */
    LastElement = frame->DataElements - 1;
    if (LastElement > (frame->TopElement + frame->lines))
    {
      /*
       * If the last element is off the screen, and we just
       * added a line, scroll the frame down so the last
       * element (the one we just added) is at the bottom.
       */
      frame->TopElement = LastElement - frame->lines + 1;
      frame->CurrentElement = LastElement - frame->lines + 1;
    }
    else if (frame->DataElements > (frame->TopElement + frame->lines))
    {
      /*
       * The very last element of frame->data is at the bottom
       * of the frame, so scroll everything up one
       */
      ++frame->TopElement;
      ++frame->CurrentElement;
      debug("AddScrollData(): set TE = %d, CE = %d, DE = %d\n",
        frame->TopElement,
        frame->CurrentElement,
        frame->DataElements);
    }
  }
} /* AddScrollData() */

/*
CreateScrollingWindow()
  Allocate space for a ScrollingWindow structure and return a
pointer to it
*/

struct Frame *
CreateScrollingWindow()

{
  struct Frame *ptr;

  ptr = (struct Frame *) MyMalloc(sizeof(struct Frame));
  memset(ptr, 0, sizeof(struct Frame));

  ptr->bufptr = ptr->buffer;

  return (ptr);
} /* CreateScrollingWindow() */

/*
FindActiveScrollingWindow()
  Attempt to locate the active scrolling frame which is using
'win' as it's output window

Return: Frame pointer if found
        0 if not
*/

struct Frame *
FindActiveScrollingWindow(WINDOW *win)

{
  assert (win != 0);

  if (win == DisassemblyFrame->window)
    return (DisassemblyFrame);
  else if (win == DebugFrame->window)
    return (DebugFrame);
  else if (win == CommandOutputFrame->window)
    return (CommandOutputFrame);
  else if (win == RegistersFrame->window)
    return (RegistersFrame);
  else if (win == CommandInputFrame->window)
  {
    if (DisassemblyFrame->flags & W_RAISED)
      return (DisassemblyFrame);
    else if (DebugFrame->flags & W_RAISED)
      return (DebugFrame);
  }

  return (0);
} /* FindActiveScrollingWindow() */

/*
ClearScrollData()
  Free frame->data and reset window's fields
*/

void
ClearScrollData(struct Frame *frame)

{
  int ii;

  assert (frame != 0);

  if (frame->data)
  {
    for (ii = 0; ii < frame->DataElements; ++ii)
      MyFree(frame->data[ii]);

    MyFree(frame->data);
  }

  if (frame->addresses)
    MyFree(frame->addresses);

  frame->DataElements = 0;
  frame->TopElement = 0;
  frame->CurrentElement = 0;
} /* ClearScrollData() */

/*
ClearScrollWindow()
  Clear the scrolling frame specified by 'winnum'
*/

void
ClearScrollWindow(struct Frame *frame)

{
  assert(frame != 0);

  /*
   * Delete frame data
   */
  ClearScrollData(frame);

  /*
   * Redraw the frame
   */
  DrawWindow(frame);
} /* ClearScrollWindow() */

/*
HighlightLine()
 Draw the given line to the given frame in reverse (highlight)
mode.
*/

static void
HighlightLine(struct Frame *frame, char *line)

{
  char buffer[MAXLINE];

  assert(frame != 0);
  assert(frame->window != 0);
  assert(line != 0);

  wSetAttr(frame->window, aReverse, 1);

  /*
   * Pad buffer with spaces to extend the highlight across
   * the screen. Leave the last slot for the \n
   */
  memset(buffer, ' ', frame->cols);

  Strncpy(buffer, line, strlen(line));
/*  buffer[frame->cols - 1] = '\n'; */
  buffer[frame->cols] = '\0';

  /*
   * Write the line to the screen
   */
  waddstr(frame->window, buffer);

  wSetAttr(frame->window, aReverse, 0);
} /* HighlightLine() */

/*
DumpCurrentWindow()
  Dump the contents of the current window into 'fptr'

Return: 1 if successful
        0 if unsuccessful
*/

int
DumpCurrentWindow(FILE *fptr)

{
  struct Frame *frame;
  int ii;

  assert(fptr != 0);
  assert(CurrentFrame != 0);

  frame = FindActiveScrollingWindow(CurrentFrame->window);
  if (!frame)
  {
    Print(mainWorkspace_p,
          P_ERROR,
          "DumpCurrentWindow: No window selected (help window)");
    return 0;
  }

  for (ii = 0; ii < frame->DataElements; ++ii)
    fprintf(fptr, "%s\n", frame->data[ii]);

  return 1;
} /* DumpCurrentWindow() */

#endif /* USE_CURSES */
