/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: set.c,v 1.13 2002/06/29 14:16:18 cosine Exp $
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "command.h"
#include "load.h"
#include "main.h"
#include "misc.h"
#include "msg.h"
#include "print.h"
#include "set.h"

#include "libDebug.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static void setArgs(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setEntryPoint(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setFileOffset(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setOutput(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setPausePrint(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setPrompt(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin);
static void setStepDisplayRegs(struct aldWorkspace *ws, int ac, char **av,
                               unsigned int pwin);
static void setStepDisplayFpRegs(struct aldWorkspace *ws, int ac, char **av,
                                 unsigned int pwin);
static void setStepDisplayMmxRegs(struct aldWorkspace *ws, int ac, char **av,
                                  unsigned int pwin);

struct Command setcmds[] = {
  { "args", setArgs, 0 },
  { "entry-point", setEntryPoint, 0 },
  { "file-offset", setFileOffset, 0 },
  { "offset", setFileOffset, C_ALIAS },
  { "output", setOutput, 0 },
  { "pause-print", setPausePrint, 0 },
  { "prompt", setPrompt, 0 },
  { "step-display-regs", setStepDisplayRegs, 0 },
  { "step-display-fpregs", setStepDisplayFpRegs, 0 },
  { "step-display-mmxregs", setStepDisplayMmxRegs, 0 },
  { 0, 0, 0 }
};

/*
 * List of setting variables and their default values
 */

/*
DisplaySettings()
  Output a list of all settings and their values
*/

void
DisplaySettings(struct aldWorkspace *ws)

{
  struct Command *cptr;

  for (cptr = setcmds; cptr->cmd != 0; ++cptr)
  {
    if (cptr->flags & C_ALIAS)
      continue; /* don't display aliases */

    RawPrint(mainWorkspace_p, P_COMMAND, "%-30s", cptr->cmd);

    (*cptr->funcptr)(ws, 0, 0, P_COMMAND);

    RawPrint(mainWorkspace_p, P_COMMAND, "\n");
  }
} /* DisplaySettings() */

/*
setArgs()
  Set the runtime arguments for the debugged process
*/

static void
setArgs(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  char str[MAXLINE];
  char *tmp;
  int ii;
  int len;

  if (pwin != 0)
  {
    tmp = getArgsDebug(ws->debugWorkspace_p);
    RawPrint(mainWorkspace_p, pwin, "%s", tmp ? tmp : "none");
    return;
  }

  if (ac < 3)
  {
    /*
     * No arguments means unset previous args
     */
    setArgsDebug(ws->debugWorkspace_p, 0);
    return;
  }

  len = sizeof(str);

  tmp = str;
  for (ii = 2; ii < ac; ++ii)
  {
    tmp += Snprintf(tmp, len, "%s ", av[ii]);
    len = sizeof(str) - (int) (tmp - str);
  }

  setArgsDebug(ws->debugWorkspace_p, str);
} /* setArgs() */

/*
setEntryPoint()
  Sets the entry point for the program being debugged
*/

static void
setEntryPoint(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  long address;

  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "0x%08X (0x%08X)",
             getLoadedEntryPoint(ws->loadWorkspace_p),
             getLoadedVirtualEntryPoint(ws->loadWorkspace_p));
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set entry-point <address>");
    return;
  }

  address = strtol(av[2], 0, 16);

  setLoadedEntryPoint(ws->loadWorkspace_p, (unsigned int) address);
} /* setEntryPoint() */

/*
setFileOffset()
  Sets the offset from the beginning of the file being debugged.
Commands such as "disassemble" will use this value as their
starting point
*/

static void
setFileOffset(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  long address;

  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "0x%08X (0x%08X)",
             getLoadedFileOffset(ws->loadWorkspace_p),
             getLoadedVirtualFileOffset(ws->loadWorkspace_p));
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set file-offset <address>");
    return;
  }

  address = strtol(av[2], 0, 16);

  setLoadedFileOffset(ws->loadWorkspace_p, (unsigned int) address);
} /* setFileOffset() */

/*
setOutput()
  Sets the path of a file to write the window's contents to, in addition to
writing to the window
*/

static void
setOutput(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  FILE *fp;

  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "%s",
             ws->printWorkspace_p->filename ? ws->printWorkspace_p->filename : "none");
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set output <filename>");
    return;
  }

  fp = fopen(av[2], "w");
  if (!fp)
  {
    Print(mainWorkspace_p,
          P_COMMAND,
          "Unable to open file %s: %s",
          av[2],
          strerror(errno));
    return;
  }

  ws->printWorkspace_p->file_p = fp;
  ws->printWorkspace_p->filename = Strdup(av[2]);
} /* setOutput() */

/*
setPausePrint()
  Enables/Disables pausing of print bursts for commands which output
a lot of information
*/

static void
setPausePrint(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "%s",
             ws->printWorkspace_p->PausePrint ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set pause-print <on | off>");
    return;
  }

  ws->printWorkspace_p->PausePrint = StrToBool(av[2]);
} /* setPausePrint() */

/*
setPrompt()
  Set the command prompt string
*/

static void
setPrompt(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "\"%s\"",
             ws->commandWorkspace_p->CmdPrompt);
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set prompt <new prompt>");
    return;
  }

  MyFree(ws->commandWorkspace_p->CmdPrompt);
  ws->commandWorkspace_p->CmdPrompt = Strdup(av[2]);
} /* setPrompt() */

/*
setStepDisplayRegs()
  Display register contents after a single step
*/

static void
setStepDisplayRegs(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "%s",
             IsSetStepDisplayRegs(ws) ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set step-display-regs <on | off>");
    return;
  }

  if (StrToBool(av[2]))
  {
    SetStepDisplayRegs(ws);
    ws->stepDisplayFlags |= DB_REGFL_DISPLAY_GENERAL;
  }
  else
  {
    UnsetStepDisplayRegs(ws);
    ws->stepDisplayFlags &= ~DB_REGFL_DISPLAY_GENERAL;
  }
} /* setStepDisplayRegs() */

/*
setStepDisplayFpRegs()
  Display floating point register contents after a single step
*/

static void
setStepDisplayFpRegs(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "%s",
             IsSetStepDisplayFpRegs(ws) ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set step-display-fpregs <on | off>");
    return;
  }

  if (StrToBool(av[2]))
  {
    SetStepDisplayFpRegs(ws);
    ws->stepDisplayFlags |= DB_REGFL_DISPLAY_FPREGS;
  }
  else
  {
    UnsetStepDisplayFpRegs(ws);
    ws->stepDisplayFlags &= ~DB_REGFL_DISPLAY_FPREGS;
  }
} /* setStepDisplayFpRegs() */

/*
setStepDisplayMmxRegs()
  Display mmx register contents after a single step
*/

static void
setStepDisplayMmxRegs(struct aldWorkspace *ws, int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(mainWorkspace_p,
             pwin,
             "%s",
             IsSetStepDisplayMmxRegs(ws) ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(mainWorkspace_p, P_COMMAND, "Syntax: set step-display-mmxregs <on | off>");
    return;
  }

  if (StrToBool(av[2]))
  {
    SetStepDisplayMmxRegs(ws);
    ws->stepDisplayFlags |= DB_REGFL_DISPLAY_MMXREGS;
  }
  else
  {
    UnsetStepDisplayMmxRegs(ws);
    ws->stepDisplayFlags &= ~DB_REGFL_DISPLAY_MMXREGS;
  }
} /* setStepDisplayMmxRegs() */
