#include <api/api_main.h>
#include <extension.h>
#include <vm.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API sections:
 *     6. Code                  :: Creating/manipulating CODE objects
 * /


/* ---------------------------------------------------------------Code */
/* TODO : rewrite these to use curr_rec like api_search.c does */
int code_new(long rva, struct code *c)
{
	struct address a;

	if (!bdb_index_find(ADDRESS_RVA, &rva, &a)) {
		return (sys_set_lasterr(4020));
	}
	if (!a.flags & ADDR_CODE) {
		a.flags ^= ADDR_DATA;
		a.flags &= ADDR_CODE;
		addr_set_flags(rva, a.flags);
	}
	/* TODO :cehck if exists? Maybe just add DB restraint */
	bdb_record_insert(CODE, c);
	return (1);
}

int code_del(long rva)
{
	void *state;
	struct address a;

	state = db_save_state();
	if (! d_keyfind( ADDRESS_RVA, &rva ) ||
			! d_recread(&a) ){
		db_restore_state(state);
		return( sys_set_lasterr( db_error() ) );
	}

	if (a.flags & ADDR_CODE) {
		/* remove CODE flag from ADDRESS */
		a.flags ^= ADDR_CODE;
		a.flags &= ADDR_DATA;
		d_recwrite(&a);
		
		/* TODO: clean up any xrefs and such here !!! */
		bdb_record_delete(CODE_RVA, &rva);
	} 
	
	db_restore_state(state);
	return (1);
}

int code_set_func(long rva, long func_id)
{
	int x = 1;
	struct code c;

	if (bdb_index_find(CODE_RVA, &rva, &c)) {
		c.func = func_id;
		bdb_record_update(CODE_RVA, &rva, &c);
	}
	return (0);
}

int code_sprintf(char *buf, char *fmt, long rva)
{
	struct address a;

	if ( bdb_index_find(ADDRESS_RVA, &rva, &a) )
		return ( asmsprintf(buf, fmt, &a) );
	return (sys_set_lasterr(4110));
}
/* defined in api_address.c, used here */
extern int sprint_op(char *buf, char *cmt, long op, int flg, int count);

int code_sprint_op(long op, unsigned int type, char *buf, int len){
	char cmt[64] = {0};
	return( sprint_op(buf, cmt, op, type, len) );
}



/* ---------------------------------------------------------ADD_EXPR */
/* these routines deal with address expressions. If an operand is of type
 * OP_ADDREXP, then its operand field references an ID in the ADDR_EXP
 * table.
 */

int addrexp_new(int scale, int index, int base, int disp, int flags)
{
	struct addr_exp a = { 0 };

	//if (! flags) return(0); /* flags should never be 0 */
	//if (! flags) printf("flags are 0!\n");; /* flags should never be 0 */
	a.scale = scale;
	a.index = index;
	a.base = base;
	a.disp = disp;
	a.flags = flags;
	//a.id = ++seq.addr_exp;
	if (bdb_record_insert(ADDR_EXP, &a))
		return (a.id);
	return (0);
}

int addrexp_equiv( int id1, int id2 ) {
	struct addr_exp e1, e2;

	if ( 	bdb_index_find( ADDR_EXP, &id1, &e1 )  &&
		bdb_index_find( ADDR_EXP, &id2, &e2 )  ) {
		if (  	e1.scale == e2.scale &&
				e1.index == e2.index &&
				e1.base == e2.base &&
				e1.disp == e2.disp && 
				e1.flags == e2.flags 	)
			return(1);
	}
	return(0);
}

int addrexp_get(int id, struct addr_exp *exp)
{
	return (bdb_index_find(ADDR_EXP, &id, exp));
}

int addrexp_format_op(int operand, int flags, char *buf, int len)
{
	union {char c; unsigned short s; unsigned int i; } u;

	/* TODO : fix sign */
	if (!operand && flags != ADDREXP_REG) {
		buf[0] = '\0';
		return (0);
	}
	
	u.i = operand;
	switch (flags) {
		case ADDREXP_REG:
			strncpy(buf, vm_get_reg_name(operand), len);
			break;
		case ADDREXP_WORD:
			if (u.s) {
				snprintf(buf, len, "0x%X", u.s);
			}
			break;
		case ADDREXP_DWORD:
			if (u.i)
				snprintf(buf, len, "0x%X", operand);
			break;
		case ADDREXP_QWORD:
			if (operand)
				snprintf(buf, len, "0x%X", operand);
			break;
		case ADDREXP_BYTE:
		default:
			if (u.c) {
				snprintf(buf, len, "0x%02hhX", u.c);
			}
	}

	return (strlen(buf));
}
int addrexp_get_str(int id, char *string, int len)
{
	struct addr_exp exp;
	char ops[32], opi[32], opb[32], opd[32];
	int sign = 0;

	if (bdb_index_find(ADDR_EXP_ID, &id, &exp)) {
		/* add NULLs for empty values */

		/* do scale */
		addrexp_format_op(exp.scale, AddrExp_ScaleType(exp.flags), ops,
				  32);
		/* do index */
		addrexp_format_op(exp.index, AddrExp_IndexType(exp.flags), opi,
				  32);
		/* do byte */
		addrexp_format_op(exp.base, AddrExp_BaseType(exp.flags), opb,
				  32);
		/* do disp */
		sign = 0;
		if ( exp.disp < 0 ) {
			exp.disp *= -1;
			sign = 1;
		}
		addrexp_format_op(exp.disp, AddrExp_DispType(exp.flags), opd,
				  32);
		/* call assembler for address expression format */
		ext_sprint_addrexp(string, len, ops, opi, opb, opd, sign);
		return (strlen(string));
	} else
		printf("not found!\n");
	return (0);
}

int code_opcmp(long op1, int flag1, long op2, int flag2)
{
	struct addr_exp *e1, *e2;

	if ((flag1 & 0xFFFF00) != flag2 & 0xFFFF00)	/* we don't care about perm */
		return (0);
	if ((flag1 & OP_TYPE_MASK) != OP_EXPR)
		return ((op1 == op2) ? 1 : 0);

	/* OK, the operand is an ADDR_EXPR, better compare it */
	/* if addr_exp id is the same, good :) */
	if ( op1 == op2 ) return(1);
	return( addrexp_equiv( op1, op2 ) );
}

int code_effect_gen(struct code *c)
{
	struct code_effect e = {0};

	/* first, get the architecture-specific side effects */
	if ( ext_gen_effect( c, &e ) ) {
		bdb_record_insert( CODE_EFFECT, &e );
		memset( &e, 0, sizeof( struct code_effect ) );
	}
	
	/* now do more mundane stuff */
	e.rva = c->rva;
	if ( c->destType & OP_W ) {
		if ( ! c->destType & OP_R ) {
			/* reg/mem_loc is clobbered */
			e.loc = c->dest;
			e.change = CEFFECT_KILL;
			e.type = c->destType &  OP_TYPE_MASK;
			e.amount = 0;
		} else {
			/* modification is dependent on instruction type */
			switch ( c->mnemType & INS_TYPE_MASK ) {
				case INS_ADD:
					e.change = CEFFECT_ADD;
					e.amount = c->src;
					break;
				case INS_SUB:
					e.change = CEFFECT_SUB;
					e.amount = c->src;
					break;
				case INS_MUL:  
					e.change = CEFFECT_MUL;
					if ( c->destType && c->srcType ) {
						/* this rules out Intel, which uses 
						 * a side effect of the instruction
						 * to represent the div result */
						e.amount = c->src;
					} else {
						/* special Intel case ;( */
						e.loc = ext_arch->reg_gen;
						e.amount = c->dest;
					}
					break;
				case INS_DIV:
					e.change = CEFFECT_DIV;
					if ( c->destType && c->srcType ) {
						e.amount = c->src;
					} else {
						e.loc = ext_arch->reg_gen;
						e.amount = c->dest;
					}
					break; 
				case INS_INC: 
					e.change = CEFFECT_ADD;
					e.amount = c->src;
					break;
				case INS_DEC:
					e.change = CEFFECT_SUB;
					e.amount = c->src;
					break;
				case INS_SHL:
					e.change = CEFFECT_MUL;
					e.amount = c->src;
					break;
				case INS_SHR:
					e.change = CEFFECT_DIV;
					e.amount = c->src;
					break;
				/* all the other modifications we care about: */
				case INS_AND: case INS_OR: case INS_XOR:
				case INS_NOT: case INS_NEG: case INS_MOV:
				case INS_MOVCC: case INS_XCHG: case INS_XCHGCC:
				case INS_ROL: case INS_ROR: 
					e.change = CEFFECT_MOD;
					e.amount = 0;
					break;
			}
		}
	} else if ( (c->destType & OP_REG) && (c->mnemType & INS_PUSH) ) {
		/* keep track of when registers are saved */
		e.change = CEFFECT_SAVE;
		e.loc = c->dest;
		e.type = OP_REG;
		e.amount = 0;
	} else if ( (c->destType & OP_REG) && (c->mnemType & INS_POP) ) {
		/* keep track of when registers are saved */
		e.change = CEFFECT_RESTORE;
		e.loc = c->dest;
		e.type = OP_REG;
		e.amount = 0;
	}

	if ( e.change )  {
		bdb_record_insert( CODE_EFFECT, &e );
	}

	/* Thus will have to be more complete... */
	return (1);
}

