/* api.c : part of The Bastard disassembly environment ========================G
 *
 * Contains  the bulk of the internal code -- specifically, the API routines
 * provided for internal and scripting use. Functions should be in the same
 * order as in api.h
 */
#include <api/api_main.h>
#include <debug.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API Sections:
 *     3. Sections            :: Creating/manipulating file SECTION objects
 * /

/* ---------------------------------------------------------------Sections */
/* TODO: check that start/end are not in existing sections */
int sec_new(char *name, long start, int size, long pa, int type)
{
	struct section section;
	int flags, ret = 1;

	if ( (type & SECTION_CODE) || (type & SECTION_EXECUTE) )
		flags = ADDR_SECTION | ADDR_CODE;
	else
		flags = ADDR_SECTION | ADDR_DATA;

	DEBUG_PrintMsg("Creating SECTION %s\n", name);
	if (d_keyfind(SECTION_NAME, &name) == S_OKAY) {
		sys_set_lasterr(4810);
		return (0);
	} else if (db_status == S_NOTFOUND) {
		addr_new(start, size, pa, flags);
		strncpy(section.name, name, 32);
		section.rva = start;
		section.size = size;
		section.flags = type;
		if (d_fillnew(SECTION, &section) != S_OKAY) {
			db_error_msg();
			ret = 0;
		}
	} else {
		DEBUG_PrintMsg("Failed to load SECTION %s\n", name);
		db_error();
		ret = 0;
	}
	disasm_env.flags |= DB_MOD;
	return (ret);
}

int sec_del(char *name)
{
	disasm_env.flags |= DB_MOD;
	if (bdb_record_delete(SECTION_NAME, name))
		return (1);
	return (sys_set_lasterr(4820));
}

int sec_get(long rva, struct section *s)
{
	struct section sec;
	int cont;

	cont = bdb_index_first(SECTION_RVA, &sec);
	while (cont) {
		if (rva >= sec.rva && rva < sec.rva + sec.size) {
			memcpy(s, &sec, sizeof (struct section));
			return (1);
		}
		cont = bdb_index_next(SECTION_RVA, &sec);
	}
	return (0);
}

long sec_get_start(char *name)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		return (s.rva);
	}
	return (sys_set_lasterr(4820));
}

long sec_get_end(char *name)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		return (s.rva + s.size - 1);
	}
	return (sys_set_lasterr(4820));
}

int sec_get_by_rva(long rva, char *buf)
{
	struct section s;

	d_keyfrst(SECTION_NAME);
	while (db_status == S_OKAY) {
		d_recread(&s);
		if (rva >= s.rva && rva < (s.rva + s.size)) {
			strncpy(buf, s.name, 32);
			return (1);
		}
	}
	return (sys_set_lasterr(4820));
}

int sec_flags(char *name)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		return (s.flags);
	}
	return (sys_set_lasterr(4820));
}

int sec_set_start(char *name, long start)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		s.rva = start;
		if (d_recwrite(&s) == S_OKAY)
			return (1);
		else
			return (db_status);
	}
	return (sys_set_lasterr(4820));
}

int sec_set_end(char *name, long end)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		s.size = end - s.rva;
		if (d_recwrite(&s) == S_OKAY)
			return (1);
		else
			return (db_status);
	}
	return (sys_set_lasterr(4820));
}

int sec_rename(char *name, char *new_name)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		strcpy(s.name, new_name);
		if (d_recwrite(&s) == S_OKAY)
			return (1);
		else
			return (db_status);
	}
	return (sys_set_lasterr(4820));
}

int sec_set_flags(char *name, int flags)
{
	struct section s;

	strcpy(s.name, name);
	if (d_keyfind(SECTION_NAME, &s.name) == S_OKAY) {
		d_recread(&s);
		s.flags = flags;
		if (d_recwrite(&s) == S_OKAY)
			return (1);
		else
			return (db_status);
	}
	return (sys_set_lasterr(4820));
}

int sec_foreach(BAPI_CALLBACK fn, void *arg)
{
	struct section s;
	int cont, *state;

	cont = bdb_table_first(SECTION, &s);
	while (cont) {
		state = db_save_state();
		(*fn) (&s, arg);
		db_restore_state( state );
		cont = bdb_table_next(SECTION, &s);
	}
	return (1);
}

int sec_foreach_addr(struct section *s, BAPI_CALLBACK fn, void *arg)
{
	struct address a;
	int cont, *state;

	cont = bdb_find_closest_next(ADDRESS_RVA, &s->rva, &a);
	while (cont && a.rva >= s->rva && a.rva < (s->rva + s->size)) {
		state = db_save_state();
		(*fn) (&a, arg);
		db_restore_state( state );
		cont = bdb_index_next(ADDRESS_RVA, &a);
	}
	return (1);
}
