
/* DB interface for the bastard + typhoon */

#include <bastard.h>
#include <bdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "db/db_query.h"
#include "db/db_def.h"


int db_type_sizeof( int type ){
	int size;

	switch ( type ) {
      	case TYPE_CHAR: case TYPE_UCHAR:
			size = sizeof( char ); break;
      	case TYPE_SHORT: case TYPE_USHORT:
			size = sizeof( short ); break;
      	case TYPE_LONG: case TYPE_ULONG:
			size = sizeof( long ); break;
		case TYPE_POINTER:
			size = sizeof( void * ); break;
		case TYPE_INT: case TYPE_UINT: default:
			size = sizeof( int ); break;
	}
	return( size );
}

int sprint_field_value(char *buf, int len, int type, void *data, int size ){
   int rv = 0;

   switch (type){
      case TYPE_CHAR:
         if ( size > 1)
            rv=snprintf(buf, len, "%s", (char *)data);
         else
            rv=snprintf(buf, len, "%c", *(char *)data);
         break;
      case TYPE_SHORT:
         rv=snprintf(buf, len, "%d", *(short *)data); break;
      case TYPE_INT:
         rv=snprintf(buf, len, "%d", *(int *)data); break;
      case TYPE_LONG:
         rv=snprintf(buf, len, "0x%08lX", *(long *)data); break;
      case TYPE_UCHAR:
         rv=snprintf(buf, len, "%c", *(unsigned char *)data); break;
      case TYPE_USHORT:
         rv=snprintf(buf, len, "%d", *(unsigned short *)data); break;
      case TYPE_UINT:
         rv=snprintf(buf, len, "%d", *(unsigned int *)data); break;
      case TYPE_ULONG:
         rv=snprintf(buf, len, "0x%08lX", *(unsigned long *)data); break;
      case TYPE_POINTER:
         rv=snprintf(buf, len, "%p", *(char **)data); break;
      }
   return(rv);
}

int db_desc_field(db_field *field){
   switch (field->type){
      case TYPE_CHAR:
         printf("\t\tchar"); break;
      case TYPE_SHORT:
         printf("\t\tshort"); break;
      case TYPE_INT:
         printf("\t\tint"); break;
      case TYPE_LONG:
         printf("\t\tlong"); break;
      case TYPE_UCHAR:
         printf("\t\tunsigned char"); break;
      case TYPE_USHORT:
         printf("\t\tunsigned short"); break;
      case TYPE_UINT:
         printf("\t\tunsigned int"); break;
      case TYPE_ULONG:
         printf("\t\tunsigned long"); break;
   }
   printf(" %s", field->name);
   if (field->size > 1) printf("[%d]", field->size);
   printf("\n");
   return(1);
}
int db_desc_index(db_index *index){
   printf("\t\t%s { %s", index->name, index->key1->name);
   if (index->key2) printf(", %s", index->key2->name);
   printf(" }\n");
   return(1);
}

int db_sprint_field( char *buf, int len, db_field *field, void *record){
   char *data = (char *) record + field->offset;
   return ( sprint_field_value(buf, len, field->type, data, field->size));
}

int db_sprint_record( char *buf, int len, char delim, char *table, void *record) {
   db_table *t;
   db_field *fld;
   char  *data;
   char *fmt_str, tmp[256];

   t = target_schema;
   while (t) {
      if (! strcmp(table, t->name)) break;
      t = t->next;
   }
   if (! t) return(0);

   fld = t->fields;
   while (fld) {
      memset(tmp, 0, 256);
      (char *)data = (char *) record + fld->offset;
      sprint_field_value(tmp, 256, fld->type, data, fld->size);
      tmp[strlen(tmp)] = delim;
      strncat(buf, tmp, len);
      
      fld = fld->next;
   }
   /* get rid of last delim */
   len = strlen(buf);
   buf[len - 1] = 0;
   len--;

   return(len);
}

int db_print_record( char *table, void *record){
   db_table *t;
   db_field *fld;
   long *rec = record, *data;
   char *fmt_str, buf[256];

   t = target_schema;
   while (t) {
      if (! strcmp(table, t->name)) break;
      t = t->next;
   }
   if (! t) return(0);

   fld = t->fields;
   while (fld) {
      (char *)data = (char *) rec + fld->offset;
      sprint_field_value(buf, 256, fld->type, data, fld->size);
      printf("%s\n", buf);
      fld = fld->next;
   }
   printf("\n");
   return(1);
}
int db_desc_table(db_table *table){
   db_field *fld;
   db_index *idx;

   if (! table ) return( 0 );
   printf("\tTable %s\n", table->name);
   printf("\t  Record:\n");
   fld = table->fields;
   while (fld) {
      db_desc_field(fld);
      fld = fld->next;
   }
   printf("\n");

   printf("\t  Indexes:\n");
   idx = table->index;
   while (idx) {
      db_desc_index(idx);
      idx = idx->next;
   }
   printf("\n");
   return(1);
}

int db_desc_db(db_schema *schema){
   db_table *current;

   if (! schema ) schema = all_schemas;

   while ( schema ) {
   	current = schema->tables;
   	while (current) {
      	db_desc_table(current);
      	current = current->next;
   	}
	schema = schema->next;
   }
   return(1);
}

int db_table_foreach(db_table *table, void *func, void *arg){
   /* return number of foreach iterations */
   return(0);
}

/* schema management */


db_schema * db_schema_by_name( char *name) {
   db_schema *s = all_schemas;
   
   while( s ) {
      if (! strcmp(s->name, name)) return(s);
      s = s->next;
   }
   return(NULL);
}



/* table management */

db_table * db_table_by_name(db_schema *s, char *name){
	db_table *t;
	
	if ( ! s ) s = all_schemas;
	t = s->tables;
	
	while ( t ) {
		if (! strcmp(t->name, name)) return(t);
		t = t->next;
	}
	return(NULL);
}

db_table * db_table_by_id(db_schema *s, int id){
	db_table *t;
	
	if ( ! s ) s = all_schemas;
	t = s->tables;
	
	while ( t ) {
		if ( id == t->id ) return(t);
		t = t->next;
	}
	return(NULL);
}

db_index * db_index_by_name(db_table *t, char *name){
	db_index *i;
		
	if (! t) return(NULL);
	i = t->index;
	
	while ( i ) {
		if (! strcmp(i->name, name)) return(i);
		i = i->next;
	}
	return(NULL);
}

db_index * db_index_by_id(db_table *t, int id){
	db_index *i;
		
	if (! t) return(NULL);
	i = t->index;
	
	while ( i ) {
		if ( id == i->id ) return(i);
		i = i->next;
	}
	return(NULL);
}

db_field * db_field_by_name(db_table *t, char *name){
	db_field *f;
	
	if (! t) return(NULL);
	f = t->fields;
	
	while ( f ) {
		if (! strcmp(f->name, name)) return(f);
		f = f->next;
	}
	return(NULL);
}

db_field * db_field_by_id(db_table *t, int id){
	db_field *f;
	
	if (! t) return(NULL);
	f = t->fields;
	
	while ( f ) {
		if ( id == f->id ) return(f);
		f = f->next;
	}
	return(NULL);
}

int db_field_from_rec( db_field *f, char *rec, char *dest ) {
	int size;
   	char *data;

	if (! dest || ! rec || !f ) 
		return(0);

   	data = (char *) rec + f->offset;
	size = db_type_sizeof( f->type );
	memcpy( dest, data, size * f->size );
	return(1);
}


int db_set_field_in_rec( db_field *f, char *rec, void *value ){
	int size;
   	char *data;

	if (! value || ! rec || !f ) 
		return(0);

   	data = (char *) rec + f->offset;
	size = db_type_sizeof( f->type );
	memcpy( data, (char *)value, size * f->size );
	return(1);
}

/* use this to get an LL for SECTION etc */
db_result * db_table_get(db_table *table) {
   /* return LL of results structs (ptr to actual structs */
	db_result *r, head = {0};
	char *data;
	int cont;
	
	r = &head;
	data = calloc(table->size, 1);	/* saves us a memcpy each time */	
	cont = bdb_index_first(table->index->id, data);

	while (cont) {
		r->next = calloc(sizeof(db_result), 1);
		r = r->next;
		r->data = data;
		r->table = table;
		
		data = calloc(table->size, 1);
		cont = bdb_index_next(table->index->id, data);
	}
	
	free(data);		/* the last calloc will be unused */
   return(head.next);
}

/* head - foot - min -max - count */
int db_table_head( db_table  *t, void *dest ) {
	if (! t ) return(0);
	return( bdb_table_first( t->id, dest ) );
}

int db_table_foot( db_table  *t, void *dest ) {
	if (! t ) return(0);
	return( bdb_table_last( t->id, dest ) );
}

int db_table_min( db_table *t, db_field  *f, void *dest ) {
	int cont;
	char *tmp;

	if (! f || ! t || ! dest) return(0);
	if ( f->index ) {
		bdb_index_first( f->index, dest );
	}
	/* not indexed: iterate through table */
	tmp = calloc(t->size, 1);
	if (! tmp ) return(0);

	cont = bdb_table_first( t->id, dest );
	while (cont) {
		cont = bdb_table_next( t->id, tmp );
		if ( cont &&  db_field_cmp( dest, tmp, f ) > 0 )
			memcpy( dest, tmp, t->size );
	}
	free(tmp);
	return(1);
}

int db_table_max( db_table *t, db_field  *f, void *dest ) {
	int cont;
	char *tmp;

	if (! f || ! t || ! dest) return(0);
	if ( f->index ) {
		bdb_index_last( f->index, dest );
	}
	
	tmp = calloc(t->size, 1);
	if (! tmp ) return(0);
	cont = bdb_table_first( t->id, dest );
	while (cont) {
		cont = bdb_table_next( t->id, tmp );
		if ( cont &&  db_field_cmp( dest, tmp, f ) < 0 )
			memcpy( dest, tmp, t->size );
	}
	free(tmp);
	return(1);
}

/* searching */
int db_field_cmp(void *rec1, void *rec2, db_field *f){
	int rv, size;
	char *c1, *c2;

	if (!f ) return(0);
	size = f->size * db_type_sizeof(f->type);
	c1 = calloc( size, 1);
	c2 = calloc( size, 1);
	if (! c1 || ! c2) return(0);

	db_field_from_rec( f, rec1, c1 );
	db_field_from_rec( f, rec2, c2 );
	if ( f->type == TYPE_CHAR && f->size > 1 ) 
		rv = strcmp( c1, c2 );
	else
		rv = memcmp( c1, c2, size );

	free( c1 );
	free( c2 );
	return(rv);
}

db_result * db_find_indexed(db_query *q){
	/* while keyfind/keynext in index, if db_field_cmp, add to results */
	return(NULL);
}

db_result * db_find_unindexed(db_query *q){
	/* for i in table */
	/*   if db_field_cmp, add to results */
	return(NULL);
}

db_result * db_select(void *query){
	return(NULL);
}
/* inserting */
/* updating */
/* deleting */



/* db_result management */
void db_result_free(db_result *r){
	db_result *i, *j;
	
	i = r;
	while (i) {
		if (i->data) free(i->data);
		j = i->next;
		free(i);
		i = j;
	}
   return;
}
void * db_result_data(db_result *r){
   return(NULL);
}
int db_result_foreach(db_result *r, void *func, void *data){
   return(0);
}
int db_result_add(db_result *list, db_result *r){
   return(0);
}
db_result * db_result_next(db_result *r){
   return(NULL);
}
db_result * db_result_prev(db_result *r){
   return(NULL);
}
db_result * db_result_head(db_result *r){
   return(NULL);
}
db_result * db_result_tail(db_result *r){
   return(NULL);
}
int db_table_dump( int id, char d_field, char d_rec, FILE *stream ) {
	struct db_table *table;
	char *rec, buf[1024];
	int cont;

	/* get table for this ID */
	table = db_table_by_id( NULL, id );
	if (! table ) return(0);
	rec = calloc( table->size, 1 );
	if (! rec ) return(0);
	if ( ! d_rec ) d_rec = '\n';
	if ( ! d_field ) d_field = '|';

	cont = bdb_table_first( id, rec );
	while ( cont ) {
		memset( buf, 0, 1024 );
		db_sprint_record( buf, 1024, d_field, table->name, rec);
		fprintf( stream, "%s%c", buf, d_rec );
		cont = bdb_table_next( id, rec );
	}
	free(rec);
	return(1);
}
long db_valtol( void *val, int type ) {
	long l;
	
	switch ( type ) {
		case TYPE_CHAR:
			l = (long) *(char *)val; break;
		case TYPE_UCHAR:
			l = (long) *(unsigned char *)val; break;
		case TYPE_SHORT:
			l = (long) *(short *)val; break;
		case TYPE_USHORT:
			l = (long) *(unsigned short *)val; break;
		case TYPE_UINT:
			l = (long) *(unsigned int *)val; break;
		case TYPE_LONG:
			l = *(long *)val; break;
		case TYPE_ULONG:
			l = (long) *(unsigned long *)val; break;
		case TYPE_POINTER:
			l = (long) *(void **)val; break;
		case TYPE_INT:
		default:
			l = (long) *(int *)val; break;
	}
	return(l);
}

unsigned long db_valtoul( void *val, int type ) {
	unsigned long l;

	switch ( type ) {
		case TYPE_CHAR:
			l = (unsigned long) *(char *)val; break;
		case TYPE_UCHAR:
			l = (unsigned long) *(unsigned char *)val; break;
		case TYPE_SHORT:
			l = (unsigned long) *(short *)val; break;
		case TYPE_USHORT:
			l = (unsigned long) *(unsigned short *)val; break;
		case TYPE_UINT:
			l = (unsigned long) *(unsigned int *)val; break;
		case TYPE_ULONG:
			l = *(unsigned long *)val; break;
		case TYPE_LONG:
			l = (unsigned long) *(long *)val; break;
		case TYPE_POINTER:
			l = (unsigned long) *(void **)val; break;
		case TYPE_INT:
		default:
			l = (unsigned long) *(int *)val; break;
	}
	return(l);

}
